<?php
/**
 * Theme custom taxonomy.
 *
 * @package rkv-child
 */

namespace RKV\Theme\Taxonomy;

/**
 * Themes taxonomy class.
 */
class Themes extends \RKV\Utilities\Taxonomy\Base {

	/**
	 * Metabox ID.
	 *
	 * @var string
	 */
	protected $metabox_id = 'theme_tax_box';

	/**
	 * Taxonomy name.
	 *
	 * @var string
	 */
	protected $taxonomy_name = '_theme';

	/**
	 * Initialize the taxonomy.
	 */
	protected function initialize_taxonomy() {
		$labels = [
			'name'              => _x( 'Themes', 'taxonomy general name', 'rkv-theme' ),
			'singular_name'     => _x( 'Theme', 'taxonomy singular name', 'rkv-theme' ),
			'search_items'      => __( 'Search Themes', 'rkv-theme' ),
			'all_items'         => __( 'All Themes', 'rkv-theme' ),
			'parent_item'       => __( 'Parent Theme', 'rkv-theme' ),
			'parent_item_colon' => __( 'Parent Theme:', 'rkv-theme' ),
			'edit_item'         => __( 'Edit Theme', 'rkv-theme' ),
			'update_item'       => __( 'Update Theme', 'rkv-theme' ),
			'add_new_item'      => __( 'Add New Theme', 'rkv-theme' ),
			'new_item_name'     => __( 'New Theme Name', 'rkv-theme' ),
			'menu_name'         => __( 'Themes', 'rkv-theme' ),
			'not_found'         => __( 'No Themes Found', 'rkv-theme' ),
		];

		$this->taxonomy_args = [
			'labels'        => $labels,
			'hierarchical'  => true,
			'rewrite'       => false,
			'show_in_menu'  => true,
			'show_tagcloud' => false,
		];

		$this->taxonomy_post_types = [ 'post', 'rkv_podcast', 'video' ];

		add_action( 'created_' . $this->taxonomy_name, [ $this, 'maybe_create_theme_page' ] );
		add_action( 'pre_delete_term', [ $this, 'maybe_delete_theme_page' ], 10, 2 );
		add_filter( 'get_edit_term_link', [ $this, 'maybe_update_term_link' ], 10, 3 );
	}

	/**

	 * Adds a theme page when the term is created.
	 * 
	 * @param int $term_id The term ID.
	 */
	public function maybe_create_theme_page( $term_id ) {
		$term = get_term_by( 'term_id', $term_id, $this->taxonomy_name );

		// Don't create a top level theme page for child terms.
		if ( $term->parent ) {
			return;
		}

		$theme = rkv_get_theme_page( $term_id );

		if ( $theme && ! is_wp_error( $theme ) ) {
			return;
		}

		$args = [
			'post_title'  => $term->name,
			'post_type'   => 'theme',
			'post_name'   => $term->slug,
			'post_status' => 'publish',
		];

		wp_insert_post( $args );
	}

	/**
	 * Removes the theme page when the term is deleted.
	 * 
	 * @param int    $term_id  The term ID.
	 * @param string $taxonomy The term taxonomy.
	 */
	public function maybe_delete_theme_page( $term_id, $taxonomy ) {
		if ( $this->taxonomy_name !== $taxonomy ) {
			return;
		}
		
		$theme = rkv_get_theme_page( $term_id );

		if ( empty( $theme ) || is_wp_error( $theme ) ) {
			return;
		}

		wp_delete_post( $theme->ID, false );
	}

	/**
	 * Filters the edit link for a term.
	 *
	 * @since 3.1.0
	 *
	 * @param string $url      The edit link.
	 * @param int    $term_id  Term ID.
	 * @param string $taxonomy Taxonomy name.
	 * 
	 * @return string
	 */
	public function maybe_update_term_link( $url, $term_id, $taxonomy ) {
		if ( $this->taxonomy_name !== $taxonomy ) {
			return $url;
		}

		$term = get_term_by( 'term_id', $term_id, $this->taxonomy_name );

		// Don't change link for child terms.
		if ( $term->parent ) {
			return $url;
		}

		$theme = rkv_get_theme_page( $term_id );

		if ( empty( $theme ) || is_wp_error( $theme ) ) {
			return $url;
		}

		return get_edit_post_link( $theme->ID );
	}
}
