<?php
/**
 * Template helper functions.
 *
 * @package rkv
 */

/**
 * Get theme page by theme term slug or ID.
 *
 * @param mixed $slug_or_id The slug or ID.
 * @return WP_Post|bool
 */
function rkv_get_theme_page( $slug_or_id ) {
	$slug = is_numeric( $slug_or_id ) ? rkv_get_theme_template_slug( $slug_or_id ) : $slug_or_id;

	if ( empty( $slug ) ) {
		return false;
	}

	$theme = get_page_by_path( $slug, \OBJECT, 'theme' );  // phpcs:ignore

	if ( empty( $theme ) || is_wp_error( $theme ) ) {
		return false;
	}

	return $theme;
}

/**
 * Gets the theme term slug from the ID.
 *
 * @param int $term_id The term ID.
 * @return string|bool
 */
function rkv_get_theme_template_slug( $term_id ) {
	$term = get_term_by( 'ID', $term_id, '_theme' );

	if ( empty( $term ) || is_wp_error( $term ) ) {
		return false;
	}

	return $term->slug;
}

/**
 * Get theme term by theme page slug or ID.
 *
 * @param mixed $slug_or_id The slug or ID.
 * @return WP_Term|bool
 */
function rkv_get_theme_term( $slug_or_id ) {
	$slug = is_numeric( $slug_or_id ) ? rkv_get_theme_page_slug( $slug_or_id ) : $slug_or_id;

	if ( empty( $slug ) ) {
		return false;
	}

	$term = get_term_by( 'slug', $slug, '_theme' );

	if ( empty( $term ) || is_wp_error( $term ) ) {
		return false;
	}

	return $term;
}

/**
 * Gets the theme page slug from the ID.
 *
 * @param int $id The term ID.
 * @return string|bool
 */
function rkv_get_theme_page_slug( $id ) {
	$theme = get_post( $id );

	if ( empty( $theme ) || is_wp_error( $theme ) ) {
		return false;
	}

	return $theme->post_name;
}


/**
 * Get theme page by theme term slug or ID.
 *
 * @param mixed $slug_or_id The slug or ID.
 * @return string|bool
 */
function rkv_get_theme_permalink( $slug_or_id ) {
	$theme = rkv_get_theme_page( $slug_or_id );

	if ( empty( $theme ) || is_wp_error( $theme ) ) {
			return false;
	}

	return get_permalink( $theme );
}

/**
 * Get the associated theme for the post
 *
 * @param int|WP_Post $post Optional. Post ID or WP_Post object. Default is global $post.
 * @return WP_Term|bool
 */
function rkv_get_post_theme( $post = 0 ) {
	$post = get_post( $post );
	$id   = isset( $post->ID ) ? $post->ID : 0;

	$theme_terms = wp_get_object_terms( $id, '_theme', [ 'count' => 1 ] );
	if ( empty( $theme_terms ) ) {
		return false;
	}
	return array_values( $theme_terms )[0];
}

/**
 * Display the Theme term of the current post
 *
 * @param array $classes Array of classes. Default is 'rkv-theme-term'.
 * @param bool  $link whether or not to display as a link.
 * @return void|string
 */
function rkv_the_post_theme( $classes = [ 'rkv-theme-term' ], $link = true ) {
	if(function_exists( 'the_seo_framework' )){
		$theme_term = the_seo_framework()->get_primary_term(get_the_ID(), "_theme");
	}else{
		$theme_term = rkv_get_post_theme();
	}
	if ( ! empty( $theme_term ) ) {
		$classes_string  = implode( ' ', $classes );
		$theme_term_name = $theme_term->name;
		if ( $link ) {
			printf( '<a class="%1$s" href="%2$s">%3$s</a>', esc_attr( $classes_string ), esc_url( rkv_get_theme_permalink( $theme_term->term_id ) ), esc_html( $theme_term_name ) );
		} else {
			printf( '<span class="%1$s">%2$s</span>', esc_attr( $classes_string ), esc_html( $theme_term_name ) );
		}
	}
}
/**
 * The single taxonomy output html for the tag. Used on single post pages and on some sliders.
 *
 * @param int|WP_Post $post Optional. Post ID or WP_Post object. Default is global $post.
 * @return void
 */
function rkv_the_entry_theme( $post = 0 ) {
	$post = get_post( $post );
	$id   = isset( $post->ID ) ? $post->ID : 0;
	if ( $id > 1 ) {
		$theme_term = rkv_get_post_theme( $id );
		if ( ! empty( $theme_term ) ) {
			printf( "<a class='%s' href='%s'>%s</a>", esc_attr( 'rkv-entry-theme-term' ), esc_url( rkv_get_theme_permalink( $theme_term->term_id ) ), esc_html( $theme_term->name ) );
		}
	}
}

/**
 * Get the Read More string for a particular post or post type.
 *
 * @param int    $post_id   The post id.
 * @param string $post_type A provided post type, defaults to null.
 * @param bool   $icon      Show an SVG icon if applicable.
 * @return string The Read More string.
 */
function rkv_the_read_more_button_string( $post_id = 0, $post_type = null, $icon = true ) {
	if ( $post_id > 0 ) {
		$post_type = rkv_get_post_type( $post_id );
	}

	$svg = '';

	if ( null === $post_type ) {
		return;
	}

	switch ( $post_type ) {
		case 'rkv_podcast':
		case 'podcast':
			$svg              = rkv_theme_get_the_svg_contents_for_svg( '/assets/images/purdue-svg/font-awesome/headphones-solid.svg' );
			$read_more_string = __( 'Listen to this Episode', 'rkv' );
			break;
		case 'rkv_video':
		case 'video':
			$svg              = rkv_theme_get_the_svg_contents_for_svg( '/assets/images/purdue-svg/font-awesome/video-solid.svg' );
			$read_more_string = __( 'Watch the Video', 'rkv' );
			break;
		default:
			$read_more_string = __( 'Read More', 'rkv' );
			break;
	}


	if ( true === $icon ) {
		$read_more_string = $svg . esc_html( apply_filters( 'rkv_the_read_more_button_string', $read_more_string ) );
	}

	echo wp_kses( apply_filters( 'rkv_the_read_more_button', $read_more_string ), rkv_theme_kses_allowed_tags_for_svg() );
}

/**
 * Return an array of allowed SVGs for use with the wp_kses() function.
 *
 * @return array
 */
function rkv_theme_kses_allowed_tags_for_svg() {
	return [
		'svg'     => [
			'class'           => true,
			'id'              => true,
			'data-name'       => true,
			'aria-hidden'     => true,
			'aria-labelledby' => true,
			'role'            => true,
			'xmlns'           => true,
			'width'           => true,
			'height'          => true,
			'viewbox'         => true, // <= Must be lower case!
			'aria-label'      => true,
			'aria-label'      => true,
		],
		'defs'    => [
			"linearGradient" => true,
		],
		'def'     => [],
		'style'   => [],
		'g'       => [
			'fill'      => true,
			'transform' => true,
		],
		'title'   => [ 'title' => true ],
		'path'    => [
			'class'     => true,
			'd'         => true,
			'fill'      => true,
			'transform' => true,
		],
		'polygon' => [
			'class'     => true,
			'd'         => true,
			'fill'      => true,
			'transform' => true,
			'points'    => true,

		],
		'text' => [
			'class'     => true,
			'd'         => true,
			'fill'      => true,
			'transform' => true,
			'font-size'    => true,
			'font-family'     => true,
			'font-weight'         => true,
			'font-size'      => true,
			'letter-spacing' => true,
		],
		'tspan'    => [
			'x'     => true,
			'y'         => true,
		],
		'rect' => [
			'width'     => true,
			'height'         => true,
			'id'      => true,
			'rx' => true,
			'transform'    => true,
			'fill'     => true,
			'stroke'         => true,
			'stroke-width'      => true,
		],
		'linearGradient' => [
			'x1'     => true,
			'x2'         => true,
			'id'      => true,
			'y1'         => true,
			'y2' => true,
			'gradientUnits'    => true,
		],
		'stop'    => [
			'offset'     => true,
			'stop-color'         => true,
		],
	];
}
/**
 * Echo the retrieved SVG file contents
 *
 * @param string $svg_filepath the file path to the SVG file.
 *
 * @return void
 */
function rkv_theme_the_svg_contents_for_svg( $svg_filepath ) {
	echo rkv_theme_get_svg_contents_for_svg( $svg_filepath );
}
/**
 * Return the content of a file for use as an inline SVG file.
 *
 * @param string $svg_filepath the file path to the SVG file.
 *
 * @return string of the SVG's file content
 */
function rkv_theme_get_svg_contents_for_svg( $svg_filepath ) {
	if ( file_exists( get_stylesheet_directory() . $svg_filepath ) ) {
		return file_get_contents( get_stylesheet_directory() . $svg_filepath ); // phpcs:ignore
	}
	return '';
}

/**
 * Return the retrieved SVG file contents.
 *
 * @param string $svg_filepath the file path to the SVG file.
 *
 * @return string
 */
function rkv_theme_get_the_svg_contents_for_svg( $svg_filepath ) {
	return wp_kses( rkv_theme_get_svg_contents_for_svg( $svg_filepath ), '' );
}
/**
 * Get the article reading time. Reading time will be calculated based on 200 words read per minute.
 */
function rkv_the_article_reading_time() {
	global $post;
	if ( ! empty( $post ) ) {
		echo esc_html( sprintf( '%d Min Read', rkv_get_the_article_reading_time() ) );
	}
}

/**
 * Get the article reading time. Reading time will be calculated based on 200 words read per minute.
 *
 * @param int|WP_Post $post Optional. Post ID or WP_Post object. Default is global $post.
 * @param int         $wpm words per minute for calcuating reading time, default to 200.
 * @return string/bool The reading time.
 */
function rkv_get_the_article_reading_time( $post = 0, $wpm = 200 ) {
	$post = get_post( $post );
	if ( ! empty( $post ) ) {
		return floor( str_word_count( wp_strip_all_tags( get_the_content( '', '', $post ) ) ) / $wpm );
	}
	return false;
}

/**
 * Prints the rkv_get_media_readable_duration();
 *
 * @return void.
 */
function rkv_the_media_duration() {
	$duration = rkv_get_media_readable_duration();
	if ( ! empty( $duration ) ) {
		echo esc_attr( $duration );
	}
}

/**
 * Returns human readable duration for media, such as a podcast.
 * From a previously saved meta_key for `rkv_media_duration`
 *
 * @param int|WP_Post $post Optional. Post ID or WP_Post object. Default is global `$post`.
 * @return string/bool on Failure.
 */
function rkv_get_media_readable_duration( $post = null ) {
	$post = get_post( $post );

	if ( ! $post ) {
		return false;
	}

	if ( get_post_meta( $post->ID, 'rkv_media_duration', true ) ) {
		$media_duration = get_post_meta( $post->ID, 'rkv_media_duration', true );
		if ( $media_duration ) {
			return $media_duration;
		}
	}

	return false;
}
/**
 * Add in Fieldmanager Group for Press Package button
 *
 * @return void
 */
function rkv_add_duration_field() {
	$rkv_duration_field = new Fieldmanager_TextField(
		[
			'name'        => 'rkv_media_duration',
			'description' => __( 'The duration in HH:MM:SS or MM:SS format.', 'rkv-theme' ),
			'label'       => __( 'Media Duration', 'rkv-theme' ),
		]
	);

	$rkv_duration_field->add_meta_box( __( 'Media Duration', 'rkv-theme' ), [ 'rkv_podcast', 'post' ], 'side' );
}

add_action( 'fm_post_rkv_podcast', 'rkv_add_duration_field' );
add_action( 'fm_post_post', 'rkv_add_duration_field' );

